unit AceView;

{ ----------------------------------------------------------------
  Ace Reporter
  Copyright 1995-1998 SCT Associates, Inc.
  Written by Kevin Maher, Steve Tyrakowski
  ---------------------------------------------------------------- }

interface
{$B-}

{$define STANDALONE}
{$ifdef VER110}{$ObjExportAll On}{$endif}
uses
  {$IFDEF WIN32}
    windows,
  {$ELSE}
    winprocs,wintypes,
  {$ENDIF}
  SysUtils, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, Aceprev, ExtCtrls,
  Menus, Buttons, AceFile, Acemeter, AceSetup;

type
  TAceViewerMode = set of (vmGenerating, vmPrinting);

  TAceViewer = class(TForm)
    LeftPreview: TAcePreview;
    PrinterSetup: TPrinterSetupDialog;
    MainMenu1: TMainMenu;
    File1: TMenuItem;
    Open1: TMenuItem;
    Close1: TMenuItem;
    N1: TMenuItem;
    Save1: TMenuItem;
    N2: TMenuItem;
    PrintSetup1: TMenuItem;
    Print1: TMenuItem;
    N3: TMenuItem;
    mitemExit: TMenuItem;
    OpenFileDialog: TOpenDialog;
    SaveFileDialog: TSaveDialog;
    ToolBar: TPanel;
    sendprinter: TSpeedButton;
    btnClose: TSpeedButton;
    ZoomValues: TComboBox;
    ViewToolbar: TMenuItem;
    TwoPages: TMenuItem;
    RightPreview: TAcePreview;
    Divider: TPanel;
    CursorPosition: TPanel;
    firstpage: TSpeedButton;
    PreviousPage: TSpeedButton;
    NextPage: TSpeedButton;
    LastPage: TSpeedButton;
    PageNum: TLabel;
    Options1: TMenuItem;
    Synchronized: TMenuItem;
    TreeducerPrint1: TMenuItem;
    LoadMeter: TAceMeter;
    Stop: TSpeedButton;
    Navigate1: TMenuItem;
    Next1: TMenuItem;
    Back1: TMenuItem;
    FirstPage1: TMenuItem;
    LastPage1: TMenuItem;
    GotoPage1: TMenuItem;
    Button1: TButton;
    Button2: TButton;
    SaveDialog1: TSaveDialog;
    SaveDialog2: TSaveDialog;
    procedure Open1Click(Sender: TObject);
    procedure Save1Click(Sender: TObject);
    procedure PrintSetup1Click(Sender: TObject);
    procedure btnCloseClick(Sender: TObject);
    procedure ViewToolbarClick(Sender: TObject);
    procedure TwoPagesClick(Sender: TObject);
    procedure LeftPreviewClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure RightPreviewClick(Sender: TObject);
    procedure firstpageClick(Sender: TObject);
    procedure PreviousPageClick(Sender: TObject);
    procedure NextPageClick(Sender: TObject);
    procedure LastPageClick(Sender: TObject);
    procedure DividerMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure DividerMouseMove(Sender: TObject; Shift: TShiftState; X,
      Y: Integer);
    procedure DividerMouseUp(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure Close1Click(Sender: TObject);
    procedure ZoomValuesChange(Sender: TObject);
    procedure SynchronizedClick(Sender: TObject);
    procedure LeftPreviewVertScroll(Sender: TObject);
    procedure LeftPreviewHorzScroll(Sender: TObject);
    procedure RightPreviewVertScroll(Sender: TObject);
    procedure RightPreviewHorzScroll(Sender: TObject);
    procedure sendprinterClick(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure StopClick(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure GotoPage1Click(Sender: TObject);
    procedure ToolBarMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure sendprinterMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure Button1Click(Sender: TObject);
    procedure Button2Click(Sender: TObject);

  private
    { Private declarations }
    MouseIsDown: Boolean;
    ShowFirst: Boolean;
    FOnAbort: TNotifyEvent;
    FGenerating: Boolean;
    FViewerMode: TAceViewerMode;
    procedure SetGenerating( Value: Boolean);
  public
    { Public declarations }
    CurrentPreview: TAcePreview;
    function GetLoadPercent: Single;
    procedure SelectPreview(p: TAcePreview);
    procedure UpdatePage;
    procedure SetAceFile(AF: TAceFile);
    procedure UpdateStatus(Sender: TObject);
    property OnAbort: TNotifyEvent read FOnAbort write FOnAbort;
    property Generating: Boolean read FGenerating write SetGenerating;

    property ViewerMode: TAceViewerMode read FViewerMode write FViewerMode;
    procedure UpdateControls;
    procedure SetZoom( Zoom: TAceZoom );
  end;

var
  AceViewer: TAceViewer;
  AceViewerDir: String;

implementation

uses printers, aceout, clipbrd, acedest, acepstat, acegoto, aceutil;

{$R *.DFM}

procedure TAceViewer.Open1Click(Sender: TObject);
begin
  if OpenFileDialog.Execute then
  begin
    if LeftPreview.Filer.AceFile = RightPreview.Filer.AceFile then
    begin
      CurrentPreview.Filer.AceFile := nil;
    end;
    CurrentPreview.LoadFromFile(OpenFileDialog.FileName);
    UpdatePage;
  end;
end;

procedure TAceViewer.Save1Click(Sender: TObject);
begin
  if CurrentPreview.Filer.AceFile <> nil then
  begin
    if TAceAceFile(CurrentPreview.Filer.AceFile).FileName <> '' then
      SaveFileDialog.FileName := TAceAceFile(CurrentPreview.Filer.AceFile).FileName;
  end;

  if SaveFileDialog.Execute then
  begin
    CurrentPreview.SaveToFile(SaveFileDialog.FileName);
  end;
end;

procedure TAceViewer.PrintSetup1Click(Sender: TObject);
begin
  CurrentPreview.AcePrinterSetup.SetData;
  printersetup.execute;
  CurrentPreview.AcePrinterSetup.GetData;
end;

procedure TAceViewer.btnCloseClick(Sender: TObject);
begin
  Close;
end;

procedure TAceViewer.ViewToolbarClick(Sender: TObject);
begin
  TMenuItem(Sender).Checked := Not TMenuItem(Sender).Checked;
  ToolBar.Visible := TMenuItem(Sender).Checked;
end;

procedure TAceViewer.TwoPagesClick(Sender: TObject);
begin
  TMenuItem(Sender).Checked := Not TMenuItem(Sender).Checked;
  if TMenuItem(Sender).Checked then
  begin
    if RightPreview.Filer.AceFile = nil then
    begin
      RightPreview.LoadFromAceFile(LeftPreview.Filer.AceFile);
    end;
  end else if CurrentPreview = RightPreview then SelectPreview(LeftPreview);
  RightPreview.Width := Width div 2;
  RightPreview.visible := TMenuItem(Sender).Checked;
  Divider.Visible := TMenuItem(Sender).Checked;

end;

procedure TAceViewer.LeftPreviewClick(Sender: TObject);
begin
  SelectPreview(LeftPreview);
end;

procedure TAceViewer.FormCreate(Sender: TObject);
begin
  FGenerating := False;
  ShowFirst := True;
  SelectPreview(LeftPreview);

  {$ifdef STANDALONE} 
    {only use this code if making a standalone viewer}
    printers.printer.printerindex := -1;                         
    CurrentPreview.AcePrinterSetup.GetData;          
  {$endif}

  OpenFileDialog.InitialDir := AceViewerDir;
  SaveFileDialog.InitialDir := AceViewerDir;
end;

procedure TAceViewer.SelectPreview(p: TAcePreview);
begin
  CurrentPreview := p;
  p.borderStyle := bsSingle;
  if p = LeftPreview then RightPreview.borderstyle := bsNone
  else LeftPreview.borderstyle := bsNone;
  UpdatePage;
  ZoomValues.ItemIndex := Ord(CurrentPreview.AceZoom);
end;


procedure TAceViewer.RightPreviewClick(Sender: TObject);
begin
  SelectPreview(RightPreview);
end;

procedure TAceViewer.firstpageClick(Sender: TObject);
begin
  CurrentPreview.FirstPage;
  UpdatePage;
end;

procedure TAceViewer.PreviousPageClick(Sender: TObject);
begin
  CurrentPreview.PriorPage;
  UpdatePage;
end;

procedure TAceViewer.NextPageClick(Sender: TObject);
begin
  CurrentPreview.NextPage;
  UpdatePage;
end;

procedure TAceViewer.LastPageClick(Sender: TObject);
begin
  CurrentPreview.LastPage;
  UpdatePage;
end;

procedure TAceViewer.UpdatePage;
begin
  LoadMeter.Progress := Round(GetLoadPercent);
  PageNum.Caption := 'Page: ' + IntToStr(CurrentPreview.page) + '/' + IntToStr(CurrentPreview.PageCount);
  if ShowFirst then
  begin
    if (CurrentPreview.PageCount > 1) Or (LoadMeter.Progress = 100) then
    begin
      ShowFirst := False;
      CurrentPreview.Invalidate;
    end;
  end;
  if CurrentPreview.Description = '' then Caption := 'AceViewer'
  else Caption := CurrentPreview.Description;
  if LoadMeter.Progress = 100 then Generating := False;
end;

procedure TAceViewer.DividerMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  CursorPosition.left := divider.left + x;
  MouseIsDown := true;
  CursorPosition.Visible := true;

  CursorPosition.Align := alNone;
  LeftPreview.Align := alNone;
  RightPreview.Align := alNone;
  Divider.Align := alNone;
end;

procedure TAceViewer.DividerMouseMove(Sender: TObject; Shift: TShiftState; X,
  Y: Integer);
var
  NewSpot: Integer;
begin
  if MouseIsDown then
  begin
    NewSpot := Divider.Left + x;
    if NewSpot < 20 then NewSpot := 20
    else if NewSpot > (ClientWidth - 24) then NewSpot := (ClientWidth - 24);

    CursorPosition.left := NewSpot;
  end;
end;

procedure TAceViewer.DividerMouseUp(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  RightPreview.Width := ClientWidth - CursorPosition.left;
  CursorPosition.Align := alRight;
  MouseIsDown := False;

  CursorPosition.Visible := False;

  RightPreview.Align := alRight;
  Divider.Align := alRight;
  LeftPreview.Align := alClient;

  ZoomValuesChange( ZoomValues );
end;

procedure TAceViewer.SetAceFile(AF: TAceFile);
begin
  LoadMeter.Progress := 0;
  Generating := True;
  ShowFirst := True;
  CurrentPreview.LoadFromAceFile(AF);
  CurrentPreview.AcePrinterSetup := AF.AcePrinterSetup;
end;

procedure TAceViewer.Close1Click(Sender: TObject);
begin
  if LeftPreview.Filer.AceFile <> RightPreview.Filer.AceFile then
  begin
    CurrentPreview.Filer.Acefile.free;
  end;
  CurrentPreview.Filer.Acefile := nil;
  CurrentPreview.Invalidate;
end;

procedure TAceViewer.ZoomValuesChange(Sender: TObject);
begin
  CurrentPreview.AceZoom := TAceZoom( TComboBox(Sender).ItemIndex );

  if Synchronized.Checked then
  begin
    if CurrentPreview = LeftPreview then RightPreview.AceZoom := LeftPreview.AceZoom
    else LeftPreview.AceZoom := RightPreview.AceZoom;
  end;

end;

procedure TAceViewer.SetZoom( zoom: TAceZoom );
begin
  ZoomValues.ItemIndex := Ord(Zoom);
  ZoomValuesChange( ZoomValues );
end;

procedure TAceViewer.SynchronizedClick(Sender: TObject);
begin
  TMenuItem(Sender).Checked := Not TMenuItem(Sender).Checked;
  { make sure zoom is same when synchronized }
  if TMenuItem(Sender).Checked then
  begin
    if CurrentPreview = LeftPreview then RightPreview.Zoom := LeftPreview.Zoom
    else LeftPreview.Zoom := RightPreview.Zoom;
  end;
end;

procedure TAceViewer.LeftPreviewVertScroll(Sender: TObject);
begin
  if Synchronized.Checked And TwoPages.Checked then
    RightPreview.VertScrollBar.Position := LeftPreview.VertScrollBar.Position;
end;

procedure TAceViewer.LeftPreviewHorzScroll(Sender: TObject);
begin
  if Synchronized.Checked And TwoPages.Checked then
    RightPreview.HorzScrollBar.Position := LeftPreview.HorzScrollBar.Position;
end;

procedure TAceViewer.RightPreviewVertScroll(Sender: TObject);
begin
  if Synchronized.Checked And TwoPages.Checked then
    LeftPreview.VertScrollBar.Position := RightPreview.VertScrollBar.Position;
end;

procedure TAceViewer.RightPreviewHorzScroll(Sender: TObject);
begin
  if Synchronized.Checked And TwoPages.Checked then
    LeftPreview.HorzScrollBar.Position := RightPreview.HorzScrollBar.Position;
end;

procedure TAceViewer.sendprinterClick(Sender: TObject);
var
  pd: TAcePrintDestination;
  ps: TAcePreviewStatus;
  CopyPS: TAcePrinterSetup;
begin
  CopyPS := TAcePrinterSetup.Create;
  CopyPS.Assign(CurrentPreview.AcePrinterSetup);

  pd := TAcePrintDestination.Create(Application);
  pd.Viewer := Self;
  pd.Preview := CurrentPreview;
  pd.AcePrinterSetup := CurrentPreview.AcePrinterSetup;
  pd.AcePrinterSetup.SetData;

  try
    if (pd.ShowModal = mrOk) then
    begin
      Include(FViewerMode, vmPrinting);
      UpdateControls;
      ps := TAcePreviewStatus.Create(Self);
      try
        ps.Preview := CurrentPreview;
        ps.Show;
        CurrentPreview.IgnorePrinterSettings := Not CopyPS.IsEqual(pd.AcePrinterSetup);
        CurrentPreview.PrintStatus := ps.UpdateStatus;
        CurrentPreview.SendPagesToPrinter(pd.StartPage, pd.EndPage);
      finally
        ps.free;
      end;
    end;
  finally
    pd.free;
    CopyPS.Free;
    Exclude(FViewerMode, vmPrinting);
    UpdateControls;
  end;

end;

procedure TAceViewer.FormClose(Sender: TObject; var Action: TCloseAction);
begin

  if Generating And Assigned(FOnAbort) then
  begin
    try
      FOnAbort(Self);
    except
      Application.HandleException(Self);
    end;
  end;
  { make sure this isn't called again }
  FOnAbort := nil;
  Generating := False;


  Action := caFree;
end;

function TAceViewer.GetLoadPercent: Single;
begin
  result := 0;
  if CurrentPreview.Filer.AceFile <> nil then
    result := TAceAceFile(CurrentPreview.Filer.AceFile).PercentDone;
end;

procedure TAceViewer.StopClick(Sender: TObject);
begin
  if Assigned(FOnAbort) then
  begin
    try
      FOnAbort(Self);
    except
      Application.HandleException(Self);
    end;
  end;
  { make sure this isn't called again }
  FOnAbort := nil;

  Generating := False;
end;

procedure TAceViewer.FormDestroy(Sender: TObject);
begin
  { make sure they are not deleting the same object }
  if LeftPreview.Filer.AceFile = RightPreview.Filer.AceFile then
    RightPreview.Filer.AceFile := nil;
end;

procedure TAceViewer.SetGenerating( Value: Boolean);
begin
  if FGenerating <> Value then
  begin
    FGenerating := Value;
  end;
  stop.Enabled := FGenerating;

  if Value then Include(FViewerMode, vmGenerating)
  else Exclude(FViewerMode, vmGenerating);

  UpdateControls;
end;

procedure TAceViewer.UpdateStatus(Sender: TObject);
begin
  UpdatePage;
end;

procedure TAceViewer.UpdateControls;
begin
  File1.Enabled := FViewerMode = [];
  ToolBar.Enabled := Not (vmPrinting in FViewerMode);
end;

procedure TAceViewer.GotoPage1Click(Sender: TObject);
var
  ag: TAceGotoPage;
  pg: Integer;
begin
  ag := TAceGotoPage.Create(self);
  try
    if (ag.ShowModal = mrOk) then
    begin
      pg := StrToInt(ag.PageNum.Text);
      CurrentPreview.Page := pg;
      UpdatePage;
    end;
  finally
    ag.free;
  end;
end;

procedure TAceViewer.ToolBarMouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  if (mbLeft = Button) And (ssCtrl in Shift) then
  begin
    ShowMessage('Ace Reporter Version ' + AceVersion);
  end;

end;

procedure TAceViewer.sendprinterMouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
var
  ps: TAcePreviewStatus;
begin
  if Button = mbRight then
  begin
    try
      Include(FViewerMode, vmPrinting);
      UpdateControls;
      ps := TAcePreviewStatus.Create(Self);
      try
        ps.Preview := CurrentPreview;
        ps.Show;
        CurrentPreview.IgnorePrinterSettings := False;
        CurrentPreview.PrintStatus := ps.UpdateStatus;
        CurrentPreview.SendPagesToPrinter(0,0);
      finally
        ps.free;
      end;
    finally
      Exclude(FViewerMode, vmPrinting);
      UpdateControls;
    end;
  end;
end;


procedure TAceViewer.Button1Click(Sender: TObject);
var x    : Integer;
    aBmp : TBitmap;
    aFile  : TAceFile;
begin
  if SaveDialog2.Execute then
  begin
    aFile := LeftPreview.Filer.AceFile;
    aFile.HorzScale := 100;
    aFile.VertScale := 100;
    aBmp  := TBitmap.Create;
    aBmp.Height := Round(aFile.AcePrinterSetup.Length * Screen.PixelsPerInch);
    aBmp.Width  := Round(aFile.AcePrinterSetup.Width *  Screen.PixelsPerInch);
    LeftPreview.Filer.AceFile.PlayPage(aBmp.Canvas.Handle,LeftPreview.Page);
    try
      aBmp.SaveToFile(SaveDialog2.FileName);
    finally
      aBmp.Free;
    end;
  end;
end;

procedure TAceViewer.Button2Click(Sender: TObject);
var x    : Integer;
    aWmf : TMetafile;
    aWmfCanvas: TMetaFileCanvas;
    aFile  : TAceFile;
begin

  if SaveDialog1.Execute then
  begin
    aFile := LeftPreview.Filer.AceFile;
    aFile.HorzScale := 100;
    aFile.VertScale := 100;
    aWmf  := TMetaFile.Create;
    aWmf.Enhanced := True;
    aWmf.Height := Round(aFile.AcePrinterSetup.Length * Screen.PixelsPerInch);
    aWmf.Width  := Round(aFile.AcePrinterSetup.Width *  Screen.PixelsPerInch);
    aWmfCanvas := TMetaFileCanvas.Create(aWmf,0) ;
    LeftPreview.Filer.AceFile.PlayPage(aWmfCanvas.Handle,LeftPreview.Page);
    aWmfCanvas.Free;
    try
      aWmf.SaveToFile(SaveDialog1.FileName);
    finally
      aWmf.Free;
    end;
  end;
end;

end.

